 //
// VOLKSWAGEN VAN CONTROL FOR JACOBO MONTAÑO FERNÁNDEZ (JackVanControl)
//---------------------******--------------------------//
//
// Include:
//    - Communication with Nextion GUI using serial comunication (to be used with JackVan_V7.hmi)
//         - Use only one page for all controls.
//         - Include dimming of the screen after 10 seconds without use.
//         - Hide buttons of Manual Mode in all other modes.
//    - Read analog values from simulation potentiometers
//    - Include accelerometer MCU6050 meassure to obtain th inclination angle.
//    - Include three pressure sensors (one for compressor out, one for rear suspension, one for front suspension)
//    - Include four MOSFET board to control electrovalves.
//    . Switch the LEDs without delay
//    - Include lecture and printing on screen of van's battery voltage.
//    - Include analog 1P4T with capacitive button to adjust the mode without the LCD screen.
//    - Definition of the button action as simple digital read calling a standard function.
//    - Included control:
//         - Perform pressure corrections only if a MODE button was pushed
//         - Perform angle corrections only if MODE AUTO button was pushed
//         - Increase height only if CompRead>CompThreshold
//           
//
//
// Should include:
//    - Definition of height variables to be used with two sonar modules
//    - two TOF laser distance sensor de measure height
//    
//  
//  
//  
// Thanks to:
//    - www.randomnerdtutorials.com
//    - Arduino online open community. Thank you all!
// 
// Licensed under Creative Commons (SH-BY-NC). @AGordiGuerrero
//.
// Antonio Gordillo Guerrero
// anto@unex.es
// @AGordiGuerrero
// Smart Open Lab. Escuela Politécnica de Cáceres. Universidad de Extremadura.
// www.smartopenlab.com
// @SolEpcc
//
//--------------------******---------------------------//
#include "Wire.h" // for IMU 6050 control
#include "I2Cdev.h"
#include "MPU6050.h"
#include "Nextion.h"
#include "HardwareSerial.h"

//#define SerialDEBUG // for debuging via serial port


///////////////////////////////
// Pin definitions
//////////////////////////////
// For Nextion display we use the Serial2 ports on ESP32: RX2=16, TX2=17
// For IMU MPU6050 we use I2C protocol on ESP32 pins SDA=21, SCL=22
// For analog control we use a capacitive interrupt touch button, which is connected to GPIO27 (TouchPin 7)
uint8_t buttonPin = 27; // capacitive button pin
uint8_t switchPin = 26; // analog switch pin
uint8_t VbatPin   = 34; // pot pins
uint8_t CompPin   = 33; // pot pins
uint8_t rearPressurePin  = 32; // pot pins
uint8_t frontPressurePin = 35; // pot pins

uint8_t rearUpRelayPin     = 19;
uint8_t rearDownRelayPin   = 23;
uint8_t frontUpRelayPin    = 4;
uint8_t frontDownRelayPin  = 18;

//uint8_t ledPin = 25; // optional LED pin



///////////////////////////////
// Variable definitions
//////////////////////////////

// timing variables
int currentMillis;
int previousMillisRead;
int readInterval=10000; //in milliseconds
int controlInterval=2000; //in milliseconds
int dimInterval=10000; //in milliseconds
int delayRelay=1000; // in ms
int previousMillisControl;
int previousMillisPush; // last time a button was pushed
int previousMillisRearUpRelay;
int previousMillisRearDownRelay;
int previousMillisFrontUpRelay;
int previousMillisFrontDownRelay;
#ifdef SerialDEBUG
int previousMillisSerialPrint;
int printSerialInterval=10000; //in milliseconds
#endif

// Control variables
uint8_t letsControlPressure=0; // flag 1 when a MODE button was pushed, 0 when we finish control
uint8_t letsControlAngle=0; // flag 1 when MODE AUTO button was pushed, 0 when we finish control
uint8_t flagNewH=0; // flag 1 when a MODE button was pushed, 0 when we finish control
uint8_t flagDeflated=0; // flag 1 when Front suspension was deflated
uint8_t flagFront=0; // flag 1 when Front Pressure is correct
uint8_t flagRear=0; // flag 1 when Front Pressure is correct
uint8_t flagDim=0; // flag 1 when screen is dimmed
int tolP=10; //tolerance (in PSI) for the desired Pressure
int tolAngle=3; //tolerance (in degrees) for the desired angle
int CapThresh=30; // Threshold for capacitive button
int CompThresh=120; // Threshold for compressor pressure
int PLowerThresh=80; // Threshold for deflation pressure
//int lastsimreadPRear=14; //USED FOR SIMULATION
char buff[50];

// Battery voltage read value
float Vbat=12.0;

// pressure read values
int readPFront=100;
int readPRear=100;
int readPComp=100;

// Height read values
//int readHFront;
//int readHRear;
//#define SONAR_SLAVE_ADDRESS 0x60 // i2c address of the Arduino Pro Mini module, with two sonars attached

// Angle variables
int readAngle=-2;
MPU6050 accelgyro;
int16_t xa, ya, za;
float roll;
//float pitch;

// Pressure desired values (in centimeters) for each mode
int desPParkFront=80;
int desPParkRear=80;
int desPRoadFront=120;
int desPRoadRear=120;
int desPTrailFront=150;
int desPTrailRear=150;
int desPFront=100;
int desPRear=100;

//HardwareSerial nextion(17, 16);// Nextion TX to pin 17 and RX to pin 16 of ESP32 on Serial2
//Nextion myNextion(nextion, 9600); //create a Nextion object named myNextion using the nextion serial port @ 9600bps

// Declare your Nextion objects - Example (page id = 0, component id = 1, component name = "b0") 
// Page 0 (Main)
NexText texvalVbat   = NexText(0, 25, "texvalVbat"); 
NexText texvalRear0  = NexText(0, 15, "texvalRear"); 
NexText texvalFront0 = NexText(0, 16, "texvalFront"); 
NexText texvalComp0  = NexText(0, 17, "texvalComp"); 
NexText texvalAngle0 = NexText(0, 18, "texvalAngle"); 
NexText texMode0     = NexText(0, 10, "texMode"); 
NexText texMessage0  = NexText(0, 12, "texMessage"); 
NexText texHRear0    = NexText(0, 13, "texHRear"); 
NexText texHFront0   = NexText(0, 14, "texHFront"); 
NexButton butPark0    = NexButton(0, 4, "butPark");
NexButton butRoad0    = NexButton(0, 5, "butRoad");
NexButton butTrail0   = NexButton(0, 6, "butTrail");
NexButton butManual0  = NexButton(0, 7, "butManual");
NexButton butAuto0    = NexButton(0, 8, "butAuto");
NexButton butUpFront0  = NexButton(0, 21, "butUpFront");
NexButton butDownFront0= NexButton(0, 22, "butDownFront");
NexButton butUpRear0   = NexButton(0, 19, "butUpRear");
NexButton butDownRear0 = NexButton(0, 20, "butDownRear");


// Register a button object to the touch event list.  
NexTouch *nex_listen_list[] = {
  &butPark0,
  &butRoad0, 
  &butTrail0,
  &butManual0,
  &butAuto0,
  &butUpFront0,
  &butDownFront0,
  &butUpRear0,
  &butDownRear0,
  NULL
};


/////////// SETUP ////////////
void setup() {
  
  Serial.begin(9600); //Start Serial monitor in 9600
  Serial2.begin(9600); // Serial comunication with Nextion display using serial2 port

#ifdef SerialDEBUG
  Serial.println("Welcome to Jack Van Control serial monitor\n Starting....");  
#endif

  // Nextion definitions
  nexInit();

  sendCommand("vis texMessage,0"); // Hide texMessage at startup

  // Register the Push event callback function of the components
  butPark0.attachPush(butPark0Callback, &butPark0);
  butRoad0.attachPush(butRoad0Callback, &butRoad0);
  butTrail0.attachPush(butTrail0Callback, &butTrail0);
  butAuto0.attachPush(butAuto0Callback, &butAuto0);
  butManual0.attachPush(butManual0Callback, &butManual0);
  butUpFront0.attachPush(butUpFront0Callback, &butUpFront0);
  butDownFront0.attachPush(butDownFront0Callback, &butDownFront0);
  butUpRear0.attachPush(butUpRear0Callback, &butUpRear0);
  butDownRear0.attachPush(butDownRear0Callback, &butDownRear0);

  // IMU MCU6050 initialisation
  Wire.begin(); // 21=SDA / 22=SCL
  accelgyro.initialize();
  Serial.println("Testing device connections...");
  Serial.println(accelgyro.testConnection() ? "MPU6050 connection successful" : "MPU6050 connection failed");
  //offset obtained using Examples->MCU6050-> IMU_Zero.ino
  accelgyro.setXGyroOffset(99);
  accelgyro.setYGyroOffset(-11);
  accelgyro.setZGyroOffset(-1);
  accelgyro.setXAccelOffset(-714);
  accelgyro.setYAccelOffset(-28);
  accelgyro.setZAccelOffset(1426);

  // Relay board outputs
  pinMode(rearUpRelayPin   , OUTPUT);
  pinMode(rearDownRelayPin , OUTPUT);
  pinMode(frontUpRelayPin  , OUTPUT);
  pinMode(frontDownRelayPin, OUTPUT);

  // Capacitive touch pin interrupt definition
//  touchAttachInterrupt(T7, analogbuttonpressed, CapThresh); // T7 on GPIO27


}


/////////// LOOP ////////////
void loop() {


//////////////////////////////////////////////////////////////////////
///////////////// SCREEN DIMMING SECTION /////////////////////////////
//////////////////////////////////////////////////////////////////////
  //decreasing dim level of the screen after a while, until any button is pushed
  currentMillis = millis();  
  if ((currentMillis - previousMillisPush >= dimInterval)&&(!flagDim)){
  //start dimming loop
      uint8_t i=0;
      for (i=0; i<=200; i++){ // 200 steps from 100 till 5
          sprintf(buff, "dim=%d", 100-(100-5)*i/200);
          sendCommand(buff);
          delay(20);
          flagDim=1;
      }
   }


//////////////////////////////////////////////////////////////////////
///////////////// CAPACITIVE BUTTON READING SECTION /////////////////////////////
//////////////////////////////////////////////////////////////////////
  // Reading variables from th capacitive button
  int touchValue = touchRead(buttonPin);
  //Serial.print(touchValue);
  // check if the touchValue is below the threshold
  // if it is, set ledPin to HIGH
  if(touchValue < CapThresh){ 
    analogbuttonpressed();
  }

//////////////////////////////////////////////////////////////////////
///////////////// SENSOR READING SECTION /////////////////////////////
//////////////////////////////////////////////////////////////////////
  // Reading variables from sensors and updating Nextion Display
  currentMillis = millis();  
  if (currentMillis - previousMillisRead >= readInterval){
    previousMillisRead = currentMillis;
    read_Vbat();
    read_pressures();
//    read_heights_average();
    read_angle();   

  // Updating Nextion screen text fields
    sprintf(buff, "%.1fV", Vbat); // displaying Vbat
    texvalVbat.setText(buff);
    utoa(readPRear, buff, 10); // unsigned integer to char
    texvalRear0.setText(buff);
    utoa(readPFront, buff, 10);// unsigned integer to char
    texvalFront0.setText(buff);
    utoa(readPComp, buff, 10);// unsigned integer to char
    texvalComp0.setText(buff);
    sprintf(buff, "%d\xB0", readAngle); // displaying º symbol
    texvalAngle0.setText(buff);
    //creating strings for pressur fields
    sprintf(buff, "%d PSI", desPRear);
    texHRear0.setText(buff);
    sprintf(buff, "%d PSI", desPFront);
    texHFront0.setText(buff);
 
    
   }




//////////////////////////////////////////////////////////////////////
///////////////// CONTROL SECTION /////////////////////////////
//////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////
// controlling Pressure if needed
//////////////////////////////////////////////////////////////////////

// Only controlling each controlInterval, in milliseconds
 
currentMillis = millis();  
if (currentMillis - previousMillisControl >= controlInterval){

    previousMillisControl = currentMillis;
    
    if(letsControlPressure){
//    if(0){
  
      // Reading sensors before control
//      read_Pressures_onesample();
      read_pressures();
      
      // Controlling Rear Pressure      
      if(abs(desPRear-readPRear)>tolP){ //we have to change Rear Pressure
         flagRear=0; // flag 0 when Pressure is not correct
         if(desPRear>readPRear){//we have to increase Rear Pressure
  
             if(readPComp>CompThresh){// then we can increase Pressure
  //              readPRear++; // increase Rear Pressure for simulation
                
                if(!digitalRead(rearUpRelayPin)){ // if it is low activate it and save the change time
                    // Activate corresponding delay
                    digitalWrite(rearUpRelayPin, HIGH);
                    previousMillisRearUpRelay=millis();
                    sendCommand("texHRear.pco=0x0408"); // Dark Green
                    sendCommand("texMessage.txt=\"\""); // Blank text
                }
                else{
                    if(millis() - previousMillisRearUpRelay > delayRelay){
                        digitalWrite(rearUpRelayPin, LOW);
                    }
                }
             }
             else{// else if compRead<CompThresh we cannot increase Pressure
                digitalWrite(rearUpRelayPin, LOW);
                sendCommand("vis texMessage,1"); // Show texMessage
                sendCommand("texHRear.pco=21087"); // Light Blue
                sendCommand("texMessage.txt=\"Low compressor pressure.\""); // Change text
//                delay(2000);      
             }// end if compRead>CompThresh
         }
         else if(desPRear<readPRear){//we have to decrease Rear Pressure
   //        readPRear--; // decrease Rear Pressure for simulation
           // Activate corresponding delay

               if(!digitalRead(rearDownRelayPin)){ // if it is low, activate it and save the change time
                    // Activate corresponding delay
                    digitalWrite(rearDownRelayPin, HIGH);
                    previousMillisRearDownRelay=millis();
                    sendCommand("texHRear.pco=0xB000"); // Dark Red
                    sendCommand("texMessage.txt=\"\""); // Blank text
                }
                else{
                    if(millis() - previousMillisRearDownRelay > delayRelay){
                        digitalWrite(rearDownRelayPin, LOW);
                    }
                }
          }    
          else {
              digitalWrite(rearUpRelayPin, LOW);
              digitalWrite(rearDownRelayPin, LOW);
              flagRear=1; // Rear Pressure is correct!
              sendCommand("texHRear.pco=BLACK");
          }
      }
      
      // Controlling Front Pressure      
      if(abs(desPFront-readPFront)>tolP){ //we have to change Front Pressure
         flagFront=0; // flag 0 when Pressure is not correct
         if(desPFront>readPFront){//we have to increase Front Pressure

             if(readPComp>CompThresh){// then we can increase Pressure
  //              readPFront++; // increase Rear Pressure for simulation
                // Activate corresponding delay
                if(!digitalRead(frontUpRelayPin)){ // if it is low activate it and save the change time
                    // Activate corresponding delay
                    digitalWrite(frontUpRelayPin, HIGH);
                    previousMillisFrontUpRelay=millis();
                    sendCommand("texHFront.pco=0x0408"); // Dark Green
                    sendCommand("texMessage.txt=\"\""); // Blank text
                }
                else{
                    if(millis() - previousMillisFrontUpRelay > delayRelay){
                        digitalWrite(frontUpRelayPin, LOW);
                    }
                }
             }
             else{// else if compRead<CompThresh we cannot increase Pressure
                digitalWrite(frontUpRelayPin, LOW);
                sendCommand("vis texMessage,1"); // Show texMessage
                sendCommand("texHFront.pco=21087"); // Light Blue
                sendCommand("texMessage.txt=\"Low compressor pressure.\""); // Change text
             }// end if compRead>CompThresh
         }
         else if(desPFront<readPFront){//we have to decrease Front Pressure
    //            readPFront--; // decrease front Pressure for simulation
                // Activate corresponding delay
               if(!digitalRead(frontDownRelayPin)){ // if it is low, activate it and save the change time
                    // Activate corresponding delay
                    digitalWrite(frontDownRelayPin, HIGH);
                    previousMillisFrontDownRelay=millis();
                    sendCommand("texHFront.pco=0xB000"); // Dark Red
                    sendCommand("texMessage.txt=\"\""); // Blank text
                }
                else{
                    if(millis() - previousMillisFrontDownRelay > delayRelay){
                        digitalWrite(frontDownRelayPin, LOW);
                    }
                }
          }    
          else {
              digitalWrite(rearUpRelayPin, LOW);
              digitalWrite(rearDownRelayPin, LOW);
              flagRear=1; // Rear Pressure is correct!
              sendCommand("texHRear.pco=BLACK");
          }
      } 
 
      else {
        flagFront=1; // front Pressure is correct!
        sendCommand("texHFront.pco=BLACK"); // Dark Red
      }

    if(flagFront&&flagRear){
        sendCommand("vis texMessage,0"); // Show texMessage
        letsControlPressure=0; // everything is correct
        sendCommand("texHRear.pco=BLACK");
        sendCommand("texHFront.pco=BLACK");
    }

    // Reading sensors after control
  //  read_Pressures_onesample();
    read_angle();
    read_pressures();

  // refresh values after control
    utoa(readPRear, buff, 10); // unsigned integer to char
    texvalRear0.setText(buff);
    utoa(readPFront, buff, 10);// unsigned integer to char
    texvalFront0.setText(buff);
    utoa(readPComp, buff, 10);// unsigned integer to char
    texvalComp0.setText(buff);
    sprintf(buff, "%d\xB0", readAngle); // displaying º symbol
    texvalAngle0.setText(buff);
    //creating strings for Pressure text fields
    sprintf(buff, "%d PSI", desPRear);
    texHRear0.setText(buff);
    sprintf(buff, "%d PSI", desPFront);
    texHFront0.setText(buff);
     
  } // end if letsControlPressure


 
//////////////////////////////////////////////////////////////////////
// controlling angle if needed
//////////////////////////////////////////////////////////////////////
  if(letsControlAngle){

      // Reading sensors before control
      read_angle();
      read_pressures();

      // if we have not deflated before...
      if (flagDeflated==0){
          if(readPFront>PLowerThresh){ // deflating front suspension
               if(!digitalRead(frontDownRelayPin)){ // if it is low, activate it and save the change time
                    // Activate corresponding delay
                    digitalWrite(frontDownRelayPin, HIGH);
                    previousMillisFrontDownRelay=millis();
                    sendCommand("texHFront.pco=0xB000"); // Dark Red
                    sendCommand("texMessage.txt=\"\""); // Blank text
                }
                else{
                    if(millis() - previousMillisFrontDownRelay > delayRelay){
                        digitalWrite(frontDownRelayPin, LOW);
                    }
                }
          }
          else{
          digitalWrite(frontDownRelayPin, LOW); // closing front deflating valve
          sendCommand("texHFront.pco=BLACK"); // BLACK
          }
          
          if(readPRear>PLowerThresh){ // deflating rear suspension

                if(!digitalRead(rearDownRelayPin)){ // if it is low, activate it and save the change time
                    // Activate corresponding delay
                    digitalWrite(rearDownRelayPin, HIGH);
                    previousMillisRearDownRelay=millis();
                    sendCommand("texHRear.pco=0xB000"); // Dark Red
                    sendCommand("texMessage.txt=\"\""); // Blank text
                }
                else{
                    if(millis() - previousMillisRearDownRelay > delayRelay){
                        digitalWrite(rearDownRelayPin, LOW);
                    }
                }
          }
          else{
          digitalWrite(rearDownRelayPin, LOW); // closing front deflating valve
          sendCommand("texHRear.pco=BLACK"); // BLACK
          }
        
        //if both are deflated turn flag ON
          if((readPRear<PLowerThresh)&&(readPFront<PLowerThresh)){
              flagDeflated=1;  // deflating finished      
        }
     
      }// end if flagDeflated==0



     if (flagDeflated==1){
        
        // Reading sensors again
        read_angle();
        read_pressures();
 
     // Controlling positive angles
     // Being a positive angle one with the van's front upper than its rear
        if(readAngle>tolAngle){ // if it is positive we have to inflate rear
//                readPRear++; // increase Rear Pressure for simulation
           if(readPComp>CompThresh){// then we can increase Pressure
                 
                if(!digitalRead(rearUpRelayPin)){ // if it is low activate it and save the change time
                    // Activate corresponding delay
                    digitalWrite(rearUpRelayPin, HIGH);
                    previousMillisRearUpRelay=millis();
                    sendCommand("texHRear.pco=0x0408"); // Dark Green
                    sendCommand("texMessage.txt=\"\""); // Blank text
                }
                else{
                    if(millis() - previousMillisRearUpRelay > delayRelay){
                        digitalWrite(rearUpRelayPin, LOW);
                    }
                }
           }
           else{  // else if compRead<CompThresh we cannot increase Pressure
                 digitalWrite(rearUpRelayPin, LOW); // closing rear inflating valve
                 sendCommand("vis texMessage,1"); // Show texMessage
                 sendCommand("texHRear.pco=21087"); // Light Blue
                 sendCommand("texMessage.txt=\"Low compressor pressure.\""); // Change text
            }// end if compRead>CompThresh
        }
        else{ // if not a possitive angle I can close rear inflating valve
           digitalWrite(rearUpRelayPin, LOW); // close inflating rear valve
           sendCommand("texHRear.pco=BLACK"); // BLACK
       }
                            
      // for negative angles we increase front Pressure
      if(readAngle<tolAngle){ // if it is negative we have to inflate front
            if(readPComp>CompThresh){// then we can increase Pressure
               
                 if(!digitalRead(frontUpRelayPin)){ // if it is low activate it and save the change time
                    // Activate corresponding delay
                    digitalWrite(frontUpRelayPin, HIGH);
                    previousMillisFrontUpRelay=millis();
                    sendCommand("texHFront.pco=0x0408"); // Dark Green
                    sendCommand("texMessage.txt=\"\""); // Blank text
                }
                else{
                    if(millis() - previousMillisFrontUpRelay > delayRelay){
                        digitalWrite(frontUpRelayPin, LOW);
                    }
                }
            }
            else{  // else if compRead<CompThresh we cannot increase Pressure
               digitalWrite(frontUpRelayPin, LOW); // closing rear inflating valve
               sendCommand("vis texMessage,1"); // Show texMessage
               sendCommand("texHFront.pco=21087"); // Light Blue
               sendCommand("texMessage.txt=\"Low compressor pressure.\""); // Change text
            }// end if compRead>CompThresh
        }
      else{ // if not a possitive angle I can close rear inflating valve
           digitalWrite(frontUpRelayPin, LOW); // close inflating rear valve
           sendCommand("texHFront.pco=BLACK"); // BLACK
        }

     } // end if Deflated=1

    if(abs(readAngle)<tolAngle){
        letsControlAngle=0; // everything is correct!
    }


    // Reading sensors after control
//    read_Pressures_onesample();
    read_angle();
    read_pressures();
    
 // refresh values after control
    utoa(readPRear, buff, 10); // unsigned integer to char
    texvalRear0.setText(buff);
    utoa(readPFront, buff, 10);// unsigned integer to char
    texvalFront0.setText(buff);
    utoa(readPComp, buff, 10);// unsigned integer to char
    texvalComp0.setText(buff);
    sprintf(buff, "%d\xB0", readAngle); // displaying º symbol
    texvalAngle0.setText(buff);
    //creating strings for Pressure text fields
    sprintf(buff, "%d PSI", desPRear);
    texHRear0.setText(buff);
    sprintf(buff, "%d PSI", desPFront);
    texHFront0.setText(buff);
       
  } // end if letsControlAngle

} 

  
////////////////////////////////////////////////////////////////////////////
///////////////// SERIAL COMMUNICATION SECTION /////////////////////////////
////////////////////////////////////////////////////////////////////////////
#ifdef SerialDEBUG
  // Sending data via serial port for debug
  currentMillis = millis();  
  if (currentMillis - previousMillisSerialPrint >= printSerialInterval){
    previousMillisSerialPrint = currentMillis;
    Serial.println("Valores deseados de altura en centímetros:"); 
    Serial.print("  Delante: "); Serial.println(desPFront);
    Serial.print("  Atras:   "); Serial.println(desPRear);
   }
#endif


///////////////// NEXTION DISPLAY UPDATE SECTION /////////////////////////////
/*    // Sending data via nextion LCD screen
  currentMillis = millis();  
  if (currentMillis - previousMillisLCDPrint >= LCDprintInterval){
    previousMillisLCDPrint = currentMillis;

  
   }
*/

   /*  
   * When a Push or push event occured every time,
   * the corresponding component[right page id and component id] in touch event list will be asked.
   */
   nexLoop(nex_listen_list); 

  //delay(20); // bit of delay...
  
}  //end main loop





// Function to read battery voltage
void read_Vbat(void){
    Vbat = analogRead(VbatPin);
    // cuadratic fitting
    // f(x)=5.58E-06*x*x - 0.00664*x +6.685
    Vbat = round(5.58E-06*Vbat*Vbat - 0.00664*Vbat +6.685);
#ifdef SerialDEBUG
    Serial.print("  Valor de voltaje en batería: ");
    Serial.println(Vbat);
#endif
}


// Function to read pressure sensors
void read_pressures(void ){
    
    readPComp = analogRead(CompPin);
    // linear fitting using "MedidasPresion_20210314.ods"
    // f(x)= 0.06159*x -23.4666
    readPComp = round(0.06159*readPComp -23.4666);
 
    readPRear = analogRead(rearPressurePin);
    // linear fitting using "MedidasPresion_20210314.ods"
    // f(x)= 0.04229*x - 9.48432
    readPRear = round(0.04229*readPRear - 9.48432);

    readPFront = analogRead(frontPressurePin);
    // linear fitting using "MedidasPresion_20210314.ods"
    // f(x)= 0.04199*x - 8.6636
    readPFront = round(0.04199*readPFront - 8.6636);

#ifdef SerialDEBUG
    Serial.print("  Presiones leídas: ");
    Serial.print("    Presión del Compresor: "); Serial.println(readPComp);
    Serial.print("    Presión Delante: "); Serial.println(readPFront);
    Serial.print("    Presión Atras:   "); Serial.println(readPRear);
#endif

}



/*
// Function to read height sensors, now using two HY-SFR05 sonar sensors located on annother Arduino Pro-Mini board
// connected via i2C
void read_heights_onesample(void ){

  uint8_t i2csent=3; // variable sent to i2c slave for doing things

#ifdef SerialDEBUG
  Serial.println("Time to fast measure distances...");
  Serial.println("Master sending...");
#endif

  Wire.beginTransmission(SONAR_SLAVE_ADDRESS); // transmit to device #9
  Wire.write(i2csent);              // sends i2csent to slave
  Wire.endTransmission();    // stop transmitting
    // wait for a while until measurements finish
  delay (200);
    
#ifdef SerialDEBUG
    Serial.println("Requesting Data from Slave"); 
#endif

    Wire.requestFrom(SONAR_SLAVE_ADDRESS, 8);
    int bytes = Wire.available();

#ifdef SerialDEBUG
    Serial.print("Slave sent ");
    Serial.print(bytes);
    Serial.print(" bytes of information\n");
#endif

    readPFront = Wire.read();
    readPRear = Wire.read();

  ////////////////////////////////////////////////////
  // Simulation of readed values for debug....
  /*
    int simreadPRear = analogRead(HrearPin);
    simreadPRear = map(simreadPRear , 0 , 4095, 5, 45);
    int simreadPFront = analogRead(HfrontPin);
    simreadPFront = map(simreadPFront, 0 , 4095, 5, 45);

    if ((simreadPFront!=lastsimreadPFront)||(simreadHRear!=lastsimreadHRear)){
      flagNewH=1;
      Serial.println("New H found!"); 
      readHFront=simreadHFront;
      readHRear=simreadHRear;
    }
    else{
      Serial.println("NO new H found!"); 
      flagNewH=0;
    }
    lastsimreadHFront=simreadHFront;  
    lastsimreadHRear=simreadHRear;  
//////////////////////////

  
#ifdef SerialDEBUG
    Serial.println("Valores leídos de altura en centímetros:"); 
    Serial.print("   Delante: "); Serial.println(readHFront);
    Serial.print("   Atras:   "); Serial.println(readHRear);
#endif

}
 
// Function to read height sensors, now using two HY-SFR05 sonar sensors located on annother Arduino Pro-Mini board
// connected via i2C
void read_heights_average(void ){

  uint8_t i2csent=0; // variable sent to i2c slave for doing things

#ifdef SerialDEBUG
  Serial.println("Time to measure distances...");
  Serial.println("Master sending...");
#endif

  Wire.beginTransmission(SONAR_SLAVE_ADDRESS); // transmit to device #9
  Wire.write(i2csent);              // sends i2csent to slave
  Wire.endTransmission();    // stop transmitting
    // wait for a while until measurements finish
  delay (2000);
    
#ifdef SerialDEBUG
    Serial.println("Requesting Data from Slave"); 
#endif

    Wire.requestFrom(SONAR_SLAVE_ADDRESS, 8);
    int bytes = Wire.available();

#ifdef SerialDEBUG
    Serial.print("Slave sent ");
    Serial.print(bytes);
    Serial.print(" bytes of information\n");
#endif

    readHFront = Wire.read();
    readHRear = Wire.read();

  ////////////////////////////////////////////////////
  // Simulation of readed values for debug....
  /*
    int simreadHRear = analogRead(HrearPin);
    simreadHRear = map(simreadHRear , 0 , 4095, 5, 45);
    int simreadHFront = analogRead(HfrontPin);
    simreadHFront = map(simreadHFront, 0 , 4095, 5, 45);

    if ((simreadHFront!=lastsimreadHFront)||(simreadHRear!=lastsimreadHRear)){
      flagNewH=1;
      Serial.println("New H found!"); 
      readHFront=simreadHFront;
      readHRear=simreadHRear;
    }
    else{
      Serial.println("NO new H found!"); 
      flagNewH=0;
    }
    lastsimreadHFront=simreadHFront;  
    lastsimreadHRear=simreadHRear;  
  
//////////////////////////

  
#ifdef SerialDEBUG
    Serial.println("Valores leídos de altura en centímetros:"); 
    Serial.print("   Delante: "); Serial.println(readHFront);
    Serial.print("   Atras:   "); Serial.println(readPRear);
#endif

}
*/


// Function to read the front/rear angle
void read_angle(void ){

    
  // Estimate angle values from acceleration values from IMU using i2cdev and MCU6050 libraries
  accelgyro.getAcceleration(&xa, &ya, &za);
  roll = atan2(ya , za) * 180.0 / PI;
//  pitch = atan2(-xa , sqrt(ya * ya + za * za)) * 180.0 / PI;

  readAngle=(int)roll;

  
#ifdef SerialDEBUG
    Serial.println("Valor leídos del ángulo de inclinación:"); 
    Serial.print("roll = ");
    Serial.print(roll,1);
//    Serial.print(", pitch = ");
//    Serial.println(pitch,1);
    Serial.print("   Ángulo delante/atrás (roll): "); Serial.println(readAngle);
#endif
 } 



/// Button callbacks functions
/*
 * Button butPark0 component Push callback function. 
 * When the Parking button is released, the text on the texMode changes and global variables set. 
 */
void butPark0Callback(void *ptr) {
  // saving push time
  previousMillisPush=millis();
  flagDim=0; // to dim after a while
  letsControlPressure=1; // we have to control height
  letsControlAngle=0; // we do not control angle
  texMode0.setText("PARKING MODE");
  // Hidding manual control buttons
  sendCommand("vis butUpFront,0"); // Hide button
  sendCommand("vis butDownFront,0"); // Hide button
  sendCommand("vis butUpRear,0"); // Hide button
  sendCommand("vis butDownRear,0"); // Hide button
#ifdef DEBUG_SERIAL_ENABLE
  dbSerial.println("Parking Mode ON");
#endif
  desPFront=desPParkFront;
  desPRear=desPParkRear;
}

void butRoad0Callback(void *ptr) {
  // saving push time
  previousMillisPush=millis();
  flagDim=0; // to dim after a while
  letsControlPressure=1; // we have to control height
  letsControlAngle=0; // we do not control angle
  texMode0.setText("ROAD MODE");
  // Hidding manual control buttons
  sendCommand("vis butUpFront,0"); // Hide button
  sendCommand("vis butDownFront,0"); // Hide button
  sendCommand("vis butUpRear,0"); // Hide button
  sendCommand("vis butDownRear,0"); // Hide button
#ifdef DEBUG_SERIAL_ENABLE
  dbSerial.println("Road Mode ON");
#endif
  desPFront=desPRoadFront;
  desPRear=desPRoadRear;
}


void butAuto0Callback(void *ptr) {
  // saving push time
  previousMillisPush=millis();
  flagDim=0; // to dim after a while
  letsControlPressure=0; // we do not control height
  letsControlAngle=1; // we do control angle
  flagDeflated=0; // to start deflating
  texMode0.setText("AUTOLEVEL MODE");
  // Hidding manual control buttons
  sendCommand("vis butUpFront,0"); // Hide button
  sendCommand("vis butDownFront,0"); // Hide button
  sendCommand("vis butUpRear,0"); // Hide button
  sendCommand("vis butDownRear,0"); // Hide button
#ifdef DEBUG_SERIAL_ENABLE
  dbSerial.println("Auto Mode ON");
#endif
// in auto mode we choose the parameters of the Parking mode
  desPFront=desPParkFront;
  desPRear=desPParkRear;
}


void butTrail0Callback(void *ptr) {
  // saving push time
  previousMillisPush=millis();
  flagDim=0; // to dim after a while
  letsControlPressure=1; // we have to control height
  letsControlAngle=0; // we do not control angle
  texMode0.setText("TRAIL MODE");
  // Hidding manual control buttons
  sendCommand("vis butUpFront,0"); // Hide button
  sendCommand("vis butDownFront,0"); // Hide button
  sendCommand("vis butUpRear,0"); // Hide button
  sendCommand("vis butDownRear,0"); // Hide button
#ifdef DEBUG_SERIAL_ENABLE
  dbSerial.println("Trail Mode ON");
#endif 
  desPFront=desPTrailFront;
  desPRear=desPTrailRear;
}


void butManual0Callback(void *ptr) {
  // saving push time
  previousMillisPush=millis();
  flagDim=0; // to dim after a while
  letsControlPressure=0; // we do not control height
  letsControlAngle=0; // we do not control angle
  texMode0.setText("MANUAL MODE");
  // Showing manual control buttons
  sendCommand("vis butUpFront,1"); // Show button
  sendCommand("vis butDownFront,1"); // Show button
  sendCommand("vis butUpRear,1"); // Show button
  sendCommand("vis butDownRear,1"); // Show button

#ifdef DEBUG_SERIAL_ENABLE
  dbSerial.println("Manual Mode ON");
#endif
  desPFront=20;
  desPRear=20;

}

void butUpFront0Callback(void *ptr) {
  // saving push time
  previousMillisPush=millis();
  flagDim=0; // to dim after a while
#ifdef DEBUG_SERIAL_ENABLE
  dbSerial.println("butUpFront pushed");
#endif

  // here we have to change heigth by activating a relay
//  Serial.println("butUpFront Pushed");
   digitalWrite(frontUpRelayPin, HIGH);
   delay(delayRelay);
   digitalWrite(frontUpRelayPin, LOW);

}

void butDownFront0Callback(void *ptr) {
  // saving push time
  previousMillisPush=millis();
  flagDim=0; // to dim after a while
#ifdef DEBUG_SERIAL_ENABLE
  dbSerial.println("butDownFront pushed");
#endif

  // here we have to change heigth by activating a relay
//  Serial.println("butDownFront Pushed");
   digitalWrite(frontDownRelayPin, HIGH);
   delay(delayRelay);
   digitalWrite(frontDownRelayPin, LOW);


}


void butUpRear0Callback(void *ptr) {
  // saving push time
  previousMillisPush=millis();
  flagDim=0; // to dim after a while
#ifdef DEBUG_SERIAL_ENABLE
  dbSerial.println("butUpRear pushed");
#endif

  // here we have to change heigth by activating a relay
  //Serial.println("butUpRear Pushed");

   digitalWrite(rearUpRelayPin, HIGH);
   delay(delayRelay);
   digitalWrite(rearUpRelayPin, LOW);

}

void butDownRear0Callback(void *ptr) {
  // saving push time
  previousMillisPush=millis();
  flagDim=0; // to dim after a while
#ifdef DEBUG_SERIAL_ENABLE
  dbSerial.println("butDownRear pushed");
#endif

  // here we have to change heigth by activating a relay
  //Serial.println("butDownRear Pushed");

   digitalWrite(rearDownRelayPin, HIGH);
   delay(delayRelay);
   digitalWrite(rearDownRelayPin, LOW);

}

//void IRAM_ATTR buttonpressed() {
// Standard function to response the capacitive touch button
void analogbuttonpressed() {

  previousMillisPush=millis();
  flagDim=0; // to dim after a while
  sprintf(buff, "dim=%d", 100); //set screen bright to maximum (dim=100)
  sendCommand(buff);

  // Always hide manual control buttons on screen
  sendCommand("vis butUpFront,0"); // Hide button
  sendCommand("vis butDownFront,0"); // Hide button
  sendCommand("vis butUpRear,0"); // Hide button
  sendCommand("vis butDownRear,0"); // Hide button


  uint16_t reading = analogRead(switchPin);
  // print out the values you read:
  //sprintf(buff, "%d  \n", reading);
  //Serial.println(buff);

  // make correspondence between read value and switch position
  // it is composed of a simple voltage divider
  if(reading<800){
    Serial.println("This corresponds to position AUTO"); 

    letsControlPressure=0; // we do not control height
    letsControlAngle=1; // we do control angle
    flagDeflated=0; // to start deflating
    texMode0.setText("AUTOLEVEL MODE");
  // Hidding manual control buttons
#ifdef DEBUG_SERIAL_ENABLE
  dbSerial.println("Auto Mode ON");
#endif
// in auto mode we choose the parameters of the Parking mode
  desPFront=desPParkFront;
  desPRear=desPParkRear;
  }
  else if(reading<1800){
    Serial.println("This corresponds to position PARK"); 
    letsControlPressure=1; // we have to control height
    letsControlAngle=0; // we do not control angle
    texMode0.setText("PARK MODE");
    desPFront=desPParkFront;
    desPRear=desPParkRear;
  }
  else if(reading<3000){
    Serial.println("This corresponds to position ROAD"); 
    letsControlPressure=1; // we have to control height
    letsControlAngle=0; // we do not control angle
    texMode0.setText("ROAD MODE");
    desPFront=desPRoadFront;
    desPRear=desPTrailRear;
  }
  else {
    Serial.println("This corresponds to position TRAIL"); 
    letsControlPressure=1; // we have to control height
    letsControlAngle=0; // we do not control angle
    texMode0.setText("TRAIL MODE");
    desPFront=desPTrailFront;
    desPRear=desPTrailRear;
  }

 

}
